// modules/excursions/backend/js/routes_wizard_step2.js
(function(){
  // --- Переменные модуля ---
  let map2, poiGroup;
  let addBtn, toggleBtn, saveBtn, cancelBtn, poiListDiv, poiFormDiv, poiHintDiv;
  let currentPoi = null, langs = [], routeId;
  let manualMode = false, tourismLayer = null;

  // --- Инициализация Шага 2: POI ---
  window.initStep2Module = function(rId, latStr, lngStr){
    routeId     = rId;
    addBtn      = document.getElementById('addPoi');
    toggleBtn   = document.getElementById('toggleTourism');
    saveBtn     = document.getElementById('savePoi');
    cancelBtn   = document.getElementById('cancelPoi');
    poiListDiv  = document.getElementById('poiList');
    poiFormDiv  = document.getElementById('poiForm');
    poiHintDiv  = document.getElementById('poiHint');

    // Загрузка языков
    fetch('/modules/excursions/backend/api/languages.php')
      .then(r => r.json())
      .then(data => langs = data)
      .catch(console.error);

    // Инициализация карты
    const lat = parseFloat(latStr), lng = parseFloat(lngStr);
    const center = (isFinite(lat) && isFinite(lng)) ? [lat, lng] : [47.7261, 10.3172];
    map2 = L.map('map2').setView(center, 13);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { maxZoom: 19 }).addTo(map2);

    // Подготовка слоя «других объектов», но не добавляем сразу
    if (window.addTourismLayer) {
      tourismLayer = window.addTourismLayer(map2);
      if (map2.hasLayer(tourismLayer)) {
        map2.removeLayer(tourismLayer);
      }
    }

    // Слой для пользовательских POI
    poiGroup = new L.FeatureGroup().addTo(map2);

    // Загрузка существующих POI
    loadPois();

    // Навешиваем обработчики
    addBtn.onclick    = enterAddMode;
    saveBtn.onclick   = savePoi;
    cancelBtn.onclick = cancelPoi;
    toggleBtn.onclick = toggleTourism;
    map2.on('click',   onMapClick);
  };

  // --- Переключение отображения туристических объектов ---
  function toggleTourism(){
    if (!tourismLayer) return;
    if (map2.hasLayer(tourismLayer)) {
      map2.removeLayer(tourismLayer);
      toggleBtn.textContent = 'Показать другие объекты';
    } else {
      map2.addLayer(tourismLayer);
      toggleBtn.textContent = 'Скрыть другие объекты';
    }
  }

  // --- Загрузка и отрисовка списка POI ---
  function loadPois(){
    if (!routeId) return;
    poiGroup.clearLayers();
    poiListDiv.innerHTML = '';
    fetch(`/modules/excursions/backend/api/pois.php?route_id=${routeId}`)
      .then(r => r.ok ? r.json() : [])
      .then(list => {
        if (!Array.isArray(list)) return;
        list.forEach(poi => {
          poi.lat = parseFloat(poi.lat);
          poi.lng = parseFloat(poi.lng);
          renderPoi(poi);
        });
      })
      .catch(console.error);
  }

  // --- Отрисовать один POI ---
  function renderPoi(poi){
    // Список
    const el = document.createElement('div');
    el.textContent = `#${poi.poi_order}: (${poi.lat.toFixed(5)}, ${poi.lng.toFixed(5)})`;
    el.style.cursor = 'pointer';
    el.onclick = () => {
      currentPoi = poi;
      manualMode = false;
      poiHintDiv.innerHTML = '';
      showForm();
    };
    poiListDiv.appendChild(el);

    // Маркер на карте
    L.circleMarker([poi.lat, poi.lng], {
      radius: poi.radius / 10,
      color: '#007bff'
    }).addTo(poiGroup);
  }

  // --- Вход в режим добавления новой точки ---
  function enterAddMode(){
    currentPoi = { id: null, poi_order: 0, radius: 50, lat: null, lng: null, translations: {} };
    poiFormDiv.innerHTML = '';
    saveBtn.disabled = true;
    manualMode = false;
    poiHintDiv.innerHTML = `
      <div style="margin-bottom:8px;">
        <button id="modeMap">Клик по карте</button>
        <button id="modeManual">Ручной ввод</button>
      </div>
      <div id="manualCoords" style="display:none;margin-bottom:8px;">
        <label>Широта:<br><input type="number" step="0.000001" id="manualLat"></label><br>
        <label>Долгота:<br><input type="number" step="0.000001" id="manualLng"></label><br>
        <button id="findCoords">Найти</button>
        <button id="applyCoords">Создать точку</button>
        <button id="cancelCoords">Отмена</button>
      </div>
      <div>Режим: <strong>Клик по карте</strong></div>
    `;
    document.getElementById('modeMap').onclick      = () => switchMode(false);
    document.getElementById('modeManual').onclick   = () => switchMode(true);
    document.getElementById('findCoords').onclick   = findCoords;
    document.getElementById('applyCoords').onclick  = applyCoords;
    document.getElementById('cancelCoords').onclick = cancelCoords;
  }

  // --- Переключение режимов ввода координат ---
  function switchMode(toManual){
    manualMode = toManual;
    document.getElementById('manualCoords').style.display = toManual ? 'block' : 'none';
    poiHintDiv.querySelector('strong').textContent = toManual ? 'Ручной ввод' : 'Клик по карте';
  }

  // --- Кнопка «Найти» ---
  function findCoords(){
    const la = parseFloat(document.getElementById('manualLat').value);
    const ln = parseFloat(document.getElementById('manualLng').value);
    if (isNaN(la) || la < -90 || la > 90 || isNaN(ln) || ln < -180 || ln > 180) {
      return alert('Введите корректные координаты: широта −90…+90, долгота −180…+180');
    }
    map2.setView([la, ln], 16);
    if (currentPoi._preview) {
      currentPoi._preview.setLatLng([la, ln]);
    } else {
      currentPoi._preview = L.circleMarker([la, ln], { radius:6, color:'red' }).addTo(map2);
    }
  }

  // --- Кнопка «Создать точку» ---
  function applyCoords(){
    const la = parseFloat(document.getElementById('manualLat').value);
    const ln = parseFloat(document.getElementById('manualLng').value);
    if (isNaN(la) || la < -90 || la > 90 || isNaN(ln) || ln < -180 || ln > 180) {
      return alert('Введите корректные координаты');
    }
    if (currentPoi._preview) {
      map2.removeLayer(currentPoi._preview);
      delete currentPoi._preview;
    }
    poiHintDiv.innerHTML = '';
    currentPoi.lat = la;
    currentPoi.lng = ln;
    showForm();
  }

  // --- Кнопка «Отмена» в ручном режиме ---
  function cancelCoords(){
    if (currentPoi._preview) {
      map2.removeLayer(currentPoi._preview);
      delete currentPoi._preview;
    }
    currentPoi = null;
    poiHintDiv.innerHTML = '';
  }

  // --- Обработчик клика по карте ---
  function onMapClick(e){
    if (!currentPoi || manualMode) return;
    poiHintDiv.innerHTML = '';
    currentPoi.lat = e.latlng.lat;
    currentPoi.lng = e.latlng.lng;
    showForm();
  }

  // --- Форма редактирования POI ---
  function showForm(){
    poiFormDiv.innerHTML = '';
    saveBtn.disabled = false;

    // draggable-маркер
    if (currentPoi._marker) {
      currentPoi._marker.setLatLng([currentPoi.lat, currentPoi.lng]);
    } else {
      currentPoi._marker = L.marker([currentPoi.lat, currentPoi.lng], { draggable:true })
        .addTo(map2)
        .on('dragend', evt => {
          const p = evt.target.getLatLng();
          document.getElementById('poLat').value = p.lat.toFixed(6);
          document.getElementById('poLng').value = p.lng.toFixed(6);
        });
    }

    // базовые поля
    let html = `
      <label>Порядок<br><input type="number" id="poOrder" value="${currentPoi.poi_order}"></label>
      <label>Радиус, м<br><input type="number" id="poRadius" value="${currentPoi.radius}"></label>
      <label>Широта<br><input type="text" id="poLat" value="${currentPoi.lat.toFixed(6)}" readonly></label>
      <label>Долгота<br><input type="text" id="poLng" value="${currentPoi.lng.toFixed(6)}" readonly></label>
    `;

    // русский язык по умолчанию
    const ru = langs.find(l=>l.code==='ru') || langs[0];
    const tr0 = currentPoi.translations['ru'] || {};
    html += `
      <fieldset style="margin-top:10px;">
        <legend>${ru.name} (${ru.code})</legend>
        <label>Текст<br><textarea id="poText_ru">${tr0.text||''}</textarea></label>
        <label>Аудио<br><input type="file" id="poAudio_ru" accept="audio/*"></label>
        <label>Фото<br><input type="file" id="poImg_ru" accept="image/*"></label>
      </fieldset>
    `;

    // выбор дополнительных языков
    const others = langs.filter(l=>l.code!=='ru');
    if (others.length) {
      html += `
        <label style="margin-top:10px;">
          Добавить язык<br>
          <select id="addLang">
            <option value="">—</option>
            ${others.map(l=>`<option value="${l.code}">${l.name}</option>`).join('')}
          </select>
        </label>
      `;
    }

    poiFormDiv.innerHTML = html;

    const sel = document.getElementById('addLang');
    if (sel) sel.onchange = ()=>addLanguage(sel.value);
  }

  // --- Добавление нового языка ---
  function addLanguage(code){
    if (!code || !currentPoi) return;
    const lang = langs.find(l=>l.code===code);
    const tr  = currentPoi.translations[code] || {};
    const fs = document.createElement('fieldset');
    fs.style.marginTop = '10px';
    fs.innerHTML = `
      <legend>${lang.name} (${code})</legend>
      <label>Текст<br><textarea id="poText_${code}">${tr.text||''}</textarea></label>
      <label>Аудио<br><input type="file" id="poAudio_${code}" accept="audio/*"></label>
      <label>Фото<br><input type="file" id="poImg_${code}" accept="image/*"></label>
    `;
    document.getElementById('addLang').insertAdjacentElement('beforebegin', fs);
    document.querySelector(`#addLang option[value="${code}"]`).remove();
    document.getElementById('addLang').value = '';
  }

  // --- Сохранение POI ---
  function savePoi(){
    const form = new FormData();
    form.append('route_id', routeId);
    form.append('poi_order', document.getElementById('poOrder').value);
    form.append('lat',       document.getElementById('poLat').value);
    form.append('lng',       document.getElementById('poLng').value);
    form.append('radius',    document.getElementById('poRadius').value);

    langs.forEach(l=>{
      const ta = document.getElementById(`poText_${l.code}`);
      if (ta) form.append(`translations[${l.code}][text]`, ta.value);
      const au = document.getElementById(`poAudio_${l.code}`);
      if (au && au.files[0]) form.append(`audio_${l.code}`, au.files[0]);
      const im = document.getElementById(`poImg_${l.code}`);
      if (im && im.files[0]) form.append(`image_${l.code}`, im.files[0]);
    });

    const url = `/modules/excursions/backend/api/pois.php${currentPoi.id ? `?id=${currentPoi.id}` : ''}`;
    fetch(url, { method:'POST', body: form })
      .then(r => r.json().then(json => {
        if (r.ok) {
          alert('POI сохранён');
          currentPoi = null;
          poiFormDiv.innerHTML = '';
          poiHintDiv.innerHTML = '';
          saveBtn.disabled = true;
          loadPois();
        } else {
          alert('Ошибка: ' + (json.error || r.status));
        }
      }))
      .catch(err => {
        console.error(err);
        alert('Сетевая ошибка');
      });
  }

  // --- Отмена редактирования POI ---
  function cancelPoi(){
    currentPoi = null;
    poiFormDiv.innerHTML = '';
    poiHintDiv.innerHTML = '';
    saveBtn.disabled = true;
  }

})();  
